import { useEffect, useState } from 'react';
import { FlatList, Pressable, Text, View } from 'react-native';
import { useDispatch } from 'react-redux';
import { useSelector } from 'react-redux';
import { isArray } from 'underscore';
import RightArrowIcon from '../../../assets/svgs/refund/rightArrow.svg';
import useAuth from '../../../hooks/useAuth';
import { getMyRefunds } from '../../../redux/slices/user/myRefund/getMyRefunds';
import ProgressiveImage from '../../../src/components/ProgressiveImage';
import MyRefundSkeleton from '../../../src/skeletons/screens/profile/refund/MyRefundSkeleton';
import BackNavigation from '../../Utilities/CustomHeader/BackNavigation';
import dpr from '../../Utilities/CustomStyleAttribute/dpr';
import { ProfileStyles } from '../ProfileStyle';
import { MyRefundStyle } from './MyRefundStyle';
import usePreferences from '../../../hooks/usePreferences';
import useProcessPrice from '../../../hooks/useProcessPrice';
import { BASE_API_URL_STG } from '@env';
import { getMyOrders } from '../../../redux/slices/user/orderHistory/getOrderHistory';
import { getMyRefundWallet } from '../../../redux/slices/user/myRefund/refundWallet/getMyRefundWallet';
import CommonStyles from '../../Utilities/CommonStyles/CommonStyles';
import useLangTranslation from '../../../hooks/useLangTranslation';
import { getPreferences } from '../../../redux/slices/preferences/getPreferences';
import gapStyle from '../../Utilities/CommonStyles/gap.style';
import CustomSVG from '../../../src/components/CustomSVG/CustomSVG';

const MyRefund = (props) => {
    const { trans } = useLangTranslation();
    const orderURL = `${BASE_API_URL_STG}/user/orders?filter_status=completed`;
    const walletUrl = `${BASE_API_URL_STG}/user/wallet`;

    const {
        preference: { order_refund, default_timezone } = {},
        company: { company_phone } = {},
    } = usePreferences();
    const processPrice = useProcessPrice();
    const { myRefunds, loading: isLoading } = useSelector(
        (state) => state.getMyRefunds
    );

    const last_refund = myRefunds[myRefunds?.length - 1];
    const { access_token } = useAuth();
    const refundUrl = `${BASE_API_URL_STG}/user/refunds`;
    const dispatch = useDispatch();

    const [refundedIsNotAvailable, setRefundedIsNotAvailable] = useState(false);
    const { myOrders, loading: orderLoading } = useSelector(
        (state) => state.getMyOrdersSlice
    );

    useEffect(() => {
        const completedOrders = myOrders.filter(
            (order) => order.status == 'Completed'
        );
        const isEveryRefundedArray = [];
        completedOrders.filter((order) => {
            order.line_items.forEach((item) => {
                let totalRefunds = 0;
                item.refunds.forEach((refund) => {
                    totalRefunds += parseInt(refund.quantity);
                });
                const isEveryRefunded = item.quantity === totalRefunds;
                isEveryRefundedArray.push(isEveryRefunded);
            });
        });
        const isRefundNotAvailable = isEveryRefundedArray.every(
            (item) => item === true
        );
        setRefundedIsNotAvailable(isRefundNotAvailable);
    }, [myOrders]);

    useEffect(() => {
        handleRefresh();
    }, [dispatch]);

    const handleRefresh = () => {
        dispatch(getMyRefunds({ access_token, URL: refundUrl }));
        dispatch(getMyOrders({ access_token, URL: orderURL }));
        dispatch(getMyRefundWallet({ access_token, URL: walletUrl }));
        dispatch(getPreferences());
    };

    useEffect(() => {
        dispatch(getMyOrders({ access_token, URL: orderURL }));
    }, []);

    const [fastLoad, setFastLoad] = useState(true);
    useEffect(() => setFastLoad(false), []);
    if (fastLoad) return <View />;

    return (
        <View style={CommonStyles.cont}>
            <BackNavigation
                navigationProps={props.navigation}
                routeName={trans("my refund")}
                capitalize={false}
            />
            <View style={ProfileStyles.hrLine} />
            <FlatList
                ListHeaderComponent={() => (
                    <>
                        <View style={MyRefundStyle.container}>
                            {isLoading ? (
                                <MyRefundSkeleton />
                            ) : !isArray(myRefunds) && !isLoading ? (
                                <View style={MyRefundStyle.noRefund}>
                                    <Text style={MyRefundStyle.noRefundText}>
                                        {trans("You Have No Refund Requests Yet")}
                                    </Text>
                                </View>
                            ) : myRefunds?.length === 0 ? (
                                <View style={MyRefundStyle.noRefund}>
                                    <Text style={MyRefundStyle.noRefundText}>
                                        {trans("You Have No Refund Requests Yet")}
                                    </Text>
                                </View>
                            ) : (
                                <View>
                                    <Pressable
                                        style={MyRefundStyle.lastRefund}
                                        onPress={() =>
                                            props.navigation.navigate(
                                                'refund details',
                                                {
                                                    refundId: last_refund?.id,
                                                    timeZone: default_timezone,
                                                    contact_number:
                                                        company_phone,
                                                }
                                            )
                                        }
                                    >
                                        <View>
                                            <Text style={MyRefundStyle.title}>
                                                {trans("Last Refund")}
                                            </Text>
                                            <Text style={MyRefundStyle.text1}>
                                                {
                                                    last_refund?.line_items
                                                        ?.category
                                                }
                                            </Text>
                                            <Text
                                                style={MyRefundStyle.text2}
                                                numberOfLines={1}
                                            >
                                                {last_refund?.line_items?.name}
                                            </Text>
                                            <View
                                                style={MyRefundStyle.variations}
                                            >
                                                {last_refund?.line_items
                                                    ?.variations &&
                                                    Object.entries(
                                                        last_refund?.line_items
                                                            ?.variations
                                                    ).map((v, i) => {
                                                        return (
                                                            <View
                                                                key={i}
                                                                style={
                                                                    MyRefundStyle.variationsItem
                                                                }
                                                            >
                                                                <Text
                                                                    style={
                                                                        MyRefundStyle.attribute
                                                                    }
                                                                >
                                                                    {v[0]}:{' '}
                                                                    {v[1]}
                                                                </Text>
                                                                <View
                                                                    style={
                                                                        MyRefundStyle.hr
                                                                    }
                                                                />
                                                            </View>
                                                        );
                                                    })}
                                                <Text
                                                    style={
                                                        MyRefundStyle.attribute
                                                    }
                                                >
                                                    {trans("Qty")}:{' '}
                                                    {last_refund?.quantity_sent}
                                                </Text>
                                            </View>
                                        </View>
                                        <View>
                                            <Text
                                                style={[
                                                    MyRefundStyle.title,
                                                    gapStyle.mlAuto,
                                                ]}
                                            >
                                                {processPrice(
                                                    last_refund?.line_items
                                                        ?.price
                                                )}
                                            </Text>
                                            <View style={MyRefundStyle.imgCont}>
                                                <ProgressiveImage
                                                    source={{
                                                        uri: last_refund?.product_image_url,
                                                    }}
                                                    style={MyRefundStyle.img}
                                                />
                                            </View>
                                        </View>
                                    </Pressable>
                                    <Pressable
                                        style={MyRefundStyle.refundList}
                                        onPress={() =>
                                            props.navigation.navigate(
                                                'refund list'
                                            )
                                        }
                                        android_ripple={{ color: 'gray' }}
                                    >
                                        <Text style={MyRefundStyle.listText}>
                                           {trans("See Refund Lists")}
                                        </Text>
                                        <CustomSVG svgIcon={RightArrowIcon} isRtl={1}
                                            width={dpr(16)}
                                            height={dpr(10)}
                                            fill={'#898989'}
                                        />
                                    </Pressable>
                                </View>
                            )}
                        </View>
                    </>
                )}
                refreshing={false}
                onRefresh={handleRefresh}
            />
            {order_refund == 1 && (
                <>
                    {!refundedIsNotAvailable && (
                        <>
                            <Pressable
                                style={MyRefundStyle.requestRefund}
                                onPress={() =>
                                    props.navigation.navigate('refund')
                                }
                            >
                                <Text style={MyRefundStyle.requestText}>
                                    {trans('Request A Refund')}
                                </Text>
                            </Pressable>
                        </>
                    )}
                </>
            )}
        </View>
    );
};

export default MyRefund;
